"use client";

import React, { useEffect, useMemo, useState } from "react";
import { Skeleton } from '../components/ui/skeleton';
import { Alert, AlertDescription, AlertTitle } from '../components/ui/alert';
import { AlertCircle } from "lucide-react";
import { loadMqttLibrary, connectAndSend, MQTT_COMMANDS } from '../services/mqttService';
import { useTranslation } from 'react-i18next';
import { Card, CardHeader, CardContent, CardTitle } from '../components/ui/card';

interface GaugeChartProps {
  deviceName: string;  // This is the primary identifier, matches device_name in database
  devEUI: string;      // This is the dev_eui from devices table
  unit?: string; // 'm³' or 'kWh'
  onValveStatusChange?: (status: { icon: string; label: string } | null) => void;
  percentTextColor?: string;
  valveTextColor?: string;
}

const GaugeChart: React.FC<GaugeChartProps> = ({
  deviceName,
  devEUI,
  unit = 'm³',
  onValveStatusChange,
  percentTextColor,
  valveTextColor
}) => {
  const { t, i18n } = useTranslation();
  const [deviceData, setDeviceData] = useState<any>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const fetchDeviceData = async () => {
      setLoading(true);
      setError(null);
      try {
        const response = await fetch('/api/postgres/device-data', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ deviceName })
        });
        if (!response.ok) throw new Error('Failed to fetch device data');
        const data = await response.json();
        setDeviceData(data);
      } catch (err: any) {
        setError(err.message || 'Failed to fetch device data');
        setDeviceData(null);
      } finally {
        setLoading(false);
      }
    };
    if (deviceName) fetchDeviceData();
  }, [deviceName]);

  if (loading) return <div className="w-full h-16"><Skeleton className="w-full h-full" /></div>;
  if (error || !deviceData) return <div className="text-destructive text-sm">{error || t('gaugeChart.noDeviceData', 'No device data available')}</div>;

  const current = deviceData.current_read ?? 0;
  const max = deviceData.max_value ?? 0;
  const percent = max > 0 ? Math.min(100, (current / max) * 100) : 0;
  let barColor = '#4CAF50';
  if (percent >= 90) barColor = '#F44336';
  else if (percent >= 75) barColor = '#FF9800';
  else if (percent >= 50) barColor = '#FFEB3B';
  else if (percent >= 25) barColor = '#8BC34A';

  // valve state (с иконкой)
  const state = (deviceData.valve_state || '').trim().toLowerCase();
  const isWaterMeter = unit === 'm³';
  let valveIcon: string | null = null;
  let valveLabel = '';
  if (state === 'open valve') {
    valveIcon = '/img/open.png';
    valveLabel = isWaterMeter ? t('gaugeChart.open', 'Open') : t('gaugeChart.on', 'On');
  } else if (state === 'close valve') {
    valveIcon = '/img/close.png';
    valveLabel = isWaterMeter ? t('gaugeChart.close', 'Close') : t('gaugeChart.off', 'Off');
  } else {
    valveLabel = t('gaugeChart.noValve', 'No valve info');
  }

  // battery
  const battery = deviceData.battery_voltage !== undefined && deviceData.battery_voltage !== null
    ? `${deviceData.battery_voltage.toFixed(2)}V`
    : t('gaugeChart.noBattery', 'No battery info');

  const dir = i18n.dir();

  return (
    <Card className="w-full">
      <CardHeader>
        <CardTitle>{t(deviceData.device_type === 'Electric meter' ? 'gaugeChart.electricMeter' : 'gaugeChart.waterMeter', deviceData.device_type).toString()}: {deviceName}</CardTitle>
        <p className="text-sm text-muted-foreground">{t('gaugeChart.devEUI', 'DEVEUI').toString()}: {devEUI}</p>
      </CardHeader>
      <CardContent>
        <div className="flex flex-col gap-2 w-full" dir="ltr">
          <div className="flex items-center justify-between">
            <span className="font-bold text-2xl">{current.toFixed(2)} {unit}</span>
            <span className="text-sm text-muted-foreground">{t(deviceData.device_type === 'Electric meter' ? 'gaugeChart.electricityUsage' : 'gaugeChart.waterUsage', deviceData.device_type === 'Electric meter' ? 'Electricity Usage' : 'Water Usage').toString()}</span>
            <span className="flex items-center gap-1 font-bold" style={{ color: valveTextColor || barColor }}>
              {valveIcon && <img src={valveIcon} alt="valve" className="inline w-5 h-5 align-middle" />}
              {valveLabel}
            </span>
          </div>
          {/* Строка с процентом использования */}
          <div className="text-sm font-bold" style={{ color: percentTextColor || barColor }} dir={dir}>
            {dir === 'rtl'
              ? <>{t('gaugeChart.percentUsedLabel', 'מנוצל').toString()} <span>{Math.round(percent)}%</span></>
              : t('gaugeChart.percentUsed', '{{percent}}% used', { percent: Math.round(percent) }).toString()}
          </div>
          <div className="w-full h-3 bg-gray-200 rounded-full overflow-hidden relative" dir="ltr">
            <div
              className="h-full rounded-full transition-all absolute left-0 top-0"
              style={{ width: `${percent}%`, background: barColor }}
            />
          </div>
          {/* ticks */}
          <div className="flex flex-row justify-between text-xs text-muted-foreground w-full select-none" dir="ltr">
            <span>0%</span>
            <span>25%</span>
            <span>50%</span>
            <span>75%</span>
            <span>100%</span>
          </div>
          <div className="flex flex-wrap gap-4 mt-2 text-xs text-muted-foreground items-center" dir={dir}>
            <span>
              {t('gaugeChart.maxValue', 'Max').toString()}: <b>{max.toFixed(2)}</b> <span dir="ltr">{unit}</span>
            </span>
            <span>
              {t('gaugeChart.battery', 'Battery').toString()}: <b>{battery}</b>
            </span>
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

export default GaugeChart;
