import { NextResponse } from 'next/server';
import { postgresClient } from '../../../../lib/postgresql-client';

export async function POST(request: Request) {
  try {
    const { deviceName } = await request.json();
    console.log('Received request for device:', deviceName);

    if (!deviceName) {
      return NextResponse.json(
        { error: 'Device name is required' },
        { status: 400 }
      );
    }

    // First, get the dev_eui from devices table
    const devEuiQuery = `
      SELECT dev_eui, device_type
      FROM devices
      WHERE device_name = $1
    `;
    
    const devEuiResult = await postgresClient.query(devEuiQuery, [deviceName]);
    console.log('Device EUI result:', devEuiResult.rows);

    if (devEuiResult.rows.length === 0) {
      return NextResponse.json(
        { error: 'Device not found in devices table' },
        { status: 404 }
      );
    }

    const { dev_eui, device_type } = devEuiResult.rows[0];

    // Then get the device data from device_max_value table
    const query = `
      SELECT 
        device_name,
        COALESCE(max_value, 50) as max_value,
        COALESCE(current_read, 0) as current_read,
        COALESCE(valve_state, 'Close valve') as valve_state,
        COALESCE(battery_voltage, 0) as battery_voltage
      FROM device_max_value
      WHERE device_name = $1
    `;

    const result = await postgresClient.query(query, [deviceName]);
    console.log('Query result:', result.rows);

    if (result.rows.length === 0) {
      return NextResponse.json(
        { error: 'Device not found in device_max_value table' },
        { status: 404 }
      );
    }

    const deviceData = result.rows[0];
    console.log('Raw device data:', deviceData);

    // Transform the data to match the expected format
    const formattedData = {
      current_read: Number(deviceData.current_read),
      max_value: Number(deviceData.max_value),
      valve_state: String(deviceData.valve_state),
      battery_voltage: Number(deviceData.battery_voltage),
      dev_eui: dev_eui,
      device_type: device_type
    };

    console.log('Formatted device data:', formattedData);
    return NextResponse.json(formattedData);
  } catch (error) {
    console.error('Error fetching device data:', error);
    return NextResponse.json(
      { error: 'Failed to fetch device data', details: error instanceof Error ? error.message : 'Unknown error' },
      { status: 500 }
    );
  }
} 