import { NextRequest, NextResponse } from 'next/server';
import { postgresClient } from '../../../../lib/postgresql-client';

export async function GET(request: Request) {
  const { searchParams } = new URL(request.url);
  const p_id = searchParams.get('p_id');

  if (!p_id) {
    return NextResponse.json({ error: 'Missing p_id parameter' }, { status: 400 });
  }

  try {
    // Определяем, какой p_id использовать для получения членов семьи
    // Требования:
    // - Основной пользователь получает семью по своему p_id
    // - Добавленный пользователь получает семью по user_relationships.main_user_p_id
    let target_p_id = p_id;

    const relationshipQuery = `
      SELECT main_user_p_id, related_user_p_id
      FROM user_relationships
      WHERE related_user_p_id = $1
      LIMIT 1
    `;

    const relationshipResult = await postgresClient.query(relationshipQuery, [p_id]);
    if (relationshipResult.rows.length > 0) {
      // p_id относится к добавленному пользователю → показываем семью основного
      target_p_id = relationshipResult.rows[0].main_user_p_id.toString();
    }

    // Определяем связанного пользователя (показываем второго участника связи относительно переданного p_id)
    let relatedUser: { p_id: number; uname: string | null; relationship_type: string | null } | null = null;
    try {
      const relRowRes = await postgresClient.query(
        `SELECT main_user_p_id, related_user_p_id, relationship_type
         FROM user_relationships
         WHERE main_user_p_id = $1 OR related_user_p_id = $1
         LIMIT 1`,
        [p_id]
      );
      if (relRowRes.rows.length > 0) {
        const rel = relRowRes.rows[0];
        const providedPidNum = Number(p_id);
        const relatedPid = rel.main_user_p_id === providedPidNum
          ? rel.related_user_p_id
          : rel.main_user_p_id;
        const userRes = await postgresClient.query(
          `SELECT uname FROM users WHERE p_id = $1 LIMIT 1`,
          [relatedPid]
        );
        relatedUser = {
          p_id: relatedPid,
          uname: userRes.rows[0]?.uname ?? null,
          relationship_type: rel.relationship_type ?? null,
        };
      }
    } catch {}

    // Get family members from family_members table using target_p_id
    const query = `
      SELECT 
        id,
        p_id,
        uname,
        c_sex,
        c_name,
        c_id,
        c_age,
        created_at
      FROM family_members
      WHERE p_id = $1
      ORDER BY created_at DESC
    `;
    
    const result = await postgresClient.query(query, [target_p_id]);
    
    // Если нет записей, вернуть пустой массив, но добавить relatedUser при наличии
    if (!result.rows || result.rows.length === 0) {
      return NextResponse.json({ familyMembers: [], count: 0, relatedUser });
    }
    
    // Адаптируем данные для UI (преобразуем в формат f_name, s_name, relation)
    const adaptedFamilyMembers = result.rows.map(member => {
      // Разделяем uname на f_name и s_name
      const nameParts = (member.uname || '').split(' ');
      const f_name = nameParts[0] || '';
      const s_name = nameParts.slice(1).join(' ') || '';
      
      // Определяем relation на основе c_sex и c_age
      let relation = 'Family Member';
      if (member.c_sex === 'M') {
        relation = member.c_age && member.c_age < 18 ? 'Son' : 'Father';
      } else if (member.c_sex === 'F') {
        relation = member.c_age && member.c_age < 18 ? 'Daughter' : 'Mother';
      }
      
      return {
        id: member.id,
        f_name,
        s_name,
        relation,
        p_id: member.p_id,
        c_sex: member.c_sex,
        c_name: member.c_name,
        c_id: member.c_id,
        c_age: member.c_age,
        created_at: member.created_at
      };
    });
    
    return NextResponse.json({ 
      familyMembers: adaptedFamilyMembers,
      count: adaptedFamilyMembers.length,
      relatedUser,
    });
  } catch (error) {
    console.error('Failed to fetch family members:', error);
    
    // Если таблица не существует или другая ошибка БД, вернуть пустой массив
    if (error instanceof Error && (
      error.message.includes('relation') && error.message.includes('does not exist') ||
      error.message.includes('column') && error.message.includes('does not exist')
    )) {
      return NextResponse.json({ 
        familyMembers: [],
        count: 0
      });
    }
    
    return NextResponse.json({ 
      error: 'Failed to fetch family members', 
      details: error instanceof Error ? error.message : 'Unknown error' 
    }, { status: 500 });
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { p_id, uname, c_sex, c_name, c_id, c_age } = body;

    if (!p_id) {
      return NextResponse.json({ error: 'Missing p_id parameter' }, { status: 400 });
    }

    // Определяем, какой p_id использовать для добавления члена семьи
    // Если p_id принадлежит добавленному пользователю, добавляем запись в семью основного пользователя
    let target_p_id = p_id;

    const relationshipQuery = `
      SELECT main_user_p_id, related_user_p_id
      FROM user_relationships
      WHERE related_user_p_id = $1
      LIMIT 1
    `;

    const relationshipResult = await postgresClient.query(relationshipQuery, [p_id]);
    if (relationshipResult.rows.length > 0) {
      target_p_id = relationshipResult.rows[0].main_user_p_id.toString();
    }

    // Если uname не указан, попробуем получить данные пользователя и создать запись для него
    if (!uname) {
      try {
        // Получаем данные пользователя по p_id
        const userResponse = await fetch(`${request.nextUrl.origin}/api/postgres/users?p_id=${target_p_id}`);
        const userData = await userResponse.json();
        
        if (userData && userData.f_name && userData.s_name) {
          // Создаем uname из f_name и s_name пользователя
          const userUname = `${userData.f_name} ${userData.s_name}`.trim();
          
          // Проверяем, есть ли уже запись для этого пользователя
          const checkQuery = `
            SELECT id FROM family_members 
            WHERE p_id = $1 AND uname = $2
          `;
          const checkResult = await postgresClient.query(checkQuery, [target_p_id, userUname]);
          
          if (checkResult.rows.length === 0) {
            // Добавляем запись для самого пользователя
            const insertUserQuery = `
              INSERT INTO family_members (p_id, uname, c_sex, c_name, c_id, c_age, created_at)
              VALUES ($1, $2, $3, $4, $5, $6, NOW())
              RETURNING id, p_id, uname, c_sex, c_name, c_id, c_age, created_at
            `;
            const userValues = [target_p_id, userUname, null, null, null, null];
            await postgresClient.query(insertUserQuery, userValues);
          }
        }
      } catch (error) {
        console.error('Failed to auto-add user to family members:', error);
        // Продолжаем выполнение, даже если не удалось добавить пользователя
      }
    }

    // Insert new family member
    const query = `
      INSERT INTO family_members (p_id, uname, c_sex, c_name, c_id, c_age, created_at)
      VALUES ($1, $2, $3, $4, $5, $6, NOW())
      RETURNING id, p_id, uname, c_sex, c_name, c_id, c_age, created_at
    `;
    
    const values = [target_p_id, uname || '', c_sex || null, c_name || '', c_id || null, c_age || null];
    const result = await postgresClient.query(query, values);
    
    return NextResponse.json({ 
      success: true,
      familyMember: result.rows[0]
    });
  } catch (error) {
    console.error('Failed to create family member:', error);
    
    // Если таблица не существует, вернуть ошибку
    if (error instanceof Error && (
      error.message.includes('relation') && error.message.includes('does not exist') ||
      error.message.includes('column') && error.message.includes('does not exist')
    )) {
      return NextResponse.json({ 
        error: 'Family members table does not exist',
        details: error.message
      }, { status: 500 });
    }
    
    return NextResponse.json({ 
      error: 'Failed to create family member', 
      details: error instanceof Error ? error.message : 'Unknown error' 
    }, { status: 500 });
  }
} 