import { NextResponse } from 'next/server';
import { postgresClient } from '../../../../lib/postgresql-client';

export async function POST(request: Request) {
  try {
    const { email, fcm_token, device_name, client_device_type } = await request.json();
    if (!email || !fcm_token) {
      return NextResponse.json({ error: 'Email and FCM token are required' }, { status: 400 });
    }

    // First get the user_id from email
    const userResult = await postgresClient.query('SELECT user_id FROM users WHERE email = $1', [email]);
    if (userResult.rows.length === 0) {
      return NextResponse.json({ error: 'User not found' }, { status: 404 });
    }
    const user_id = userResult.rows[0].user_id;

    // Upsert device entry with last_seen timestamp
    await postgresClient.query(
      `INSERT INTO user_devices (user_id, fcm_token, device_name, client_device_type, last_seen)
       VALUES ($1, $2, $3, $4, NOW())
       ON CONFLICT (fcm_token) 
       DO UPDATE SET 
         last_seen = NOW(),
         device_name = COALESCE($3, user_devices.device_name),
         client_device_type = COALESCE($4, user_devices.client_device_type)`,
      [user_id, fcm_token, device_name || null, client_device_type || null]
    );

    return NextResponse.json({ success: true });
  } catch (error) {
    console.error('Failed to save FCM token:', error);
    return NextResponse.json({ 
      error: 'Failed to save FCM token',
      details: error instanceof Error ? error.message : 'Unknown error'
    }, { status: 500 });
  }
} 