import { getMessaging, getToken, onMessage } from "firebase/messaging";
import { app } from "./firebase";

let messaging: any = null;

// Check if browser supports required APIs for Firebase Messaging
const isMessagingSupported = () => {
  if (typeof window === 'undefined') return false;
  
  // Check for Service Worker support
  if (!('serviceWorker' in navigator)) return false;
  
  // Check for Push Manager support
  if (!('PushManager' in window)) return false;
  
  // Check for Notification support
  if (!('Notification' in window)) return false;
  
  return true;
};

// Initialize messaging only on client side and if supported
if (typeof window !== 'undefined' && isMessagingSupported()) {
  try {
    messaging = getMessaging(app);
  } catch (error) {
    console.error('Failed to initialize Firebase Messaging:', error);
  }
}

export const requestNotificationPermission = async () => {
  if (!messaging || !isMessagingSupported()) {
    //console.log('Firebase Messaging not supported in this browser');
    return null;
  }
  
  try {
    const permission = await Notification.requestPermission();
    if (permission === 'granted') {
      const token = await getToken(messaging, {
        vapidKey: process.env.NEXT_PUBLIC_FIREBASE_VAPID_KEY
      });
      return token;
    }
    return null;
  } catch (error) {
    console.error('Failed to get notification permission:', error);
    return null;
  }
};

export const onMessageListener = () => {
  if (!messaging || !isMessagingSupported()) {
    //console.log('Firebase Messaging not supported in this browser');
    return Promise.resolve(null);
  }
  
  try {
    return new Promise((resolve) => {
      onMessage(messaging, (payload) => {
        resolve(payload);
      });
    });
  } catch (error) {
    console.error('Failed to set up message listener:', error);
    return Promise.resolve(null);
  }
};

export { messaging }; 